// definition tooltip script. written in 2024 by quinn. https://arcanacheque.neocities.org

import {offset, shift, flip, arrow, computePosition, autoUpdate} from './floating-ui-dom@1.6.5/floating-ui.dom.esm.min.js';
// If you don't want to host your own copy of floating-ui, remove or comment 
// out the line above and uncomment the line below:
//
// import {offset, shift, flip, arrow, computePosition, autoUpdate} from 'https://cdn.jsdelivr.net/npm/@floating-ui/dom@1.6.5/+esm';

// Configurable variables for tooltip appearance/behaviour. Touch away!
const defaultPlacement = "bottom";
const tooltipOffset = 6;
const pageEdgePadding = 5;
const arrowOffset = 6;
const arrowPadding = 6;

// Sensitive code below! Touch at your own risk.
let tooltipElement, tooltipContent, tooltipStyle, referenceElement, arrowElement, cleanup = null;

setup();

function setup() {
  tooltipElement = document.createElement('div');
  tooltipContent = document.createElement('div');
  tooltipStyle = document.createElement('style');
  arrowElement = document.createElement('div'); 

  tooltipElement.appendChild(arrowElement);
  tooltipElement.appendChild(tooltipStyle);
  tooltipElement.appendChild(tooltipContent);

  tooltipElement.id = "definition-tooltip";
  tooltipElement.ariaHidden = true;

  tooltipContent.id = "definition-tooltip-content";

  arrowElement.id = "definition-tooltip-arrow";

  tooltipStyle.innerHTML = `
#definition-tooltip {
  --fade-time: 0.15s;

  position: fixed;
  opacity: 0;
  visibility: hidden;
  transition: 
    opacity var(--fade-time) ease-in-out,
    visibility var(--fade-time) ease-in-out;

  &.shown {
    visibility: visible;
    opacity: 1;
  }

  #definition-tooltip-content {
    position: relative;
    z-index: 100;
  }

  #definition-tooltip-arrow {
    position: absolute;
    z-index: 99;
  }
}
  `

  document.body.appendChild(tooltipElement);

  document.querySelectorAll("dfn").forEach((dfn) => {
    dfn.dataset.tooltip = dfn.title;
    dfn.ariaDescription ||= dfn.title
    dfn.removeAttribute('title');
    dfn.addEventListener("mouseenter", () => {
      handleShow(dfn);
    })
    dfn.addEventListener("touchstart", () => {
      handleShow(dfn);
    })
    dfn.addEventListener("mouseleave", () => {
      hideTooltip();
    })
  })

  document.addEventListener("touchstart", handleClose);
};

function updateTooltipElement() {
  if (!referenceElement) { return; }
  computePosition(referenceElement, tooltipElement, {
    strategy: 'fixed',
    placement: defaultPlacement,
    middleware: [
      offset(tooltipOffset),
      flip(),
      shift({padding: pageEdgePadding}),
      arrow({element: arrowElement, padding: arrowPadding})
    ]
  }).then(({x,y, placement, middlewareData}) => {
    // update tooltip
    Object.assign(tooltipElement.style, {
      left: `${x}px`,
      top: `${y}px`
    });

    // update arrow
    const {x: arrowX, y: arrowY} = middlewareData.arrow;
   
    const staticSide = {
      top: 'bottom',
      right: 'left',
      bottom: 'top',
      left: 'right',
    }[placement.split('-')[0]];
   
    Object.assign(arrowElement.style, {
      left: arrowX != null ? `${arrowX}px` : '',
      top: arrowY != null ? `${arrowY}px` : '',
      right: '',
      bottom: '',
      [staticSide]: `-${arrowOffset}px`,
    });
  });
};

function showTooltip() {
  tooltipElement.classList.add("shown");
  cleanup = autoUpdate(referenceElement, tooltipElement, updateTooltipElement);
  updateTooltipElement();
};

function hideTooltip() {
  if (cleanup) {
    cleanup();
    cleanup = null;
  }
  tooltipElement.classList.remove("shown");
};

function handleShow(element) {
  referenceElement = element;
  tooltipContent.innerHTML = referenceElement.dataset.tooltip;
  showTooltip();
};

function handleClose(event) { 
  if (event.target.tagName == "DFN") { return; }

  hideTooltip();
};

